// $Id: CADSREditor.cpp,v 1.6 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CADSREditor.hpp"
using Exponent::GUI::Controls::CADSREditor;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CADSREditor, CControl);

//	===========================================================================
CADSREditor::CADSREditor(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener) 
		   : CControl(root, uniqueId, area, listener)
		   , m_envelopePolygon(5)
		   , m_mouseIsOver(false)
		   , m_mouseDown(false)
		   , m_selectedPoint(e_noSelection)
		   , m_sizeOfQuarter(0)
		   , m_innerHeight(0)
		   , m_innerWidth(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CADSREditor);

	// Compute our inner area
	m_innerWidth  = m_area.getWidth()  - (CADSR_EDITOR_WINDOW_OFFSET * 2);
	m_innerHeight = m_area.getHeight() - (CADSR_EDITOR_WINDOW_OFFSET * 2);

	// Compute the size of one quarter of that area
	m_sizeOfQuarter = m_innerWidth / 4;

	// Get all the points from the envelope array
	CPoint *startPoint   = m_envelopePolygon.getPoint(0);
	CPoint *attackPoint  = m_envelopePolygon.getPoint(1);
	CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
	CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
	CPoint *releasePoint = m_envelopePolygon.getPoint(4);

	// Set all the points
	startPoint->setPoint(CADSR_EDITOR_WINDOW_OFFSET, CADSR_EDITOR_WINDOW_OFFSET + m_innerHeight);
	attackPoint->setPoint(m_sizeOfQuarter, CADSR_EDITOR_WINDOW_OFFSET);
	decayPoint->setPoint(m_sizeOfQuarter * 2, CADSR_EDITOR_WINDOW_OFFSET + (m_innerHeight / 2));
	sustainPoint->setPoint(m_sizeOfQuarter * 3, CADSR_EDITOR_WINDOW_OFFSET + (m_innerHeight / 2));
	releasePoint->setPoint(CADSR_EDITOR_WINDOW_OFFSET + m_innerWidth, CADSR_EDITOR_WINDOW_OFFSET + m_innerHeight);

	// Setup the grab handles
	m_handles[0].setRect(attackPoint->getXPosition()  - CADSR_EDITOR_HANDLE_HALF_SIZE, attackPoint->getYPosition()  - CADSR_EDITOR_HANDLE_HALF_SIZE, CADSR_EDITOR_HANDLE_SIZE, CADSR_EDITOR_HANDLE_SIZE);
	m_handles[1].setRect(decayPoint->getXPosition()   - CADSR_EDITOR_HANDLE_HALF_SIZE, decayPoint->getYPosition()   - CADSR_EDITOR_HANDLE_HALF_SIZE, CADSR_EDITOR_HANDLE_SIZE, CADSR_EDITOR_HANDLE_SIZE);
	m_handles[2].setRect(sustainPoint->getXPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE, sustainPoint->getYPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE, CADSR_EDITOR_HANDLE_SIZE, CADSR_EDITOR_HANDLE_SIZE);
	m_handles[3].setRect(releasePoint->getXPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE, releasePoint->getYPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE, CADSR_EDITOR_HANDLE_SIZE, CADSR_EDITOR_HANDLE_SIZE);

	m_values[0] = 1.0;
	m_values[1] = 1.0;
	m_values[2] = 0.5;
	m_values[3] = 1.0;

	// Reset any booleans
	m_mouseIsOver   = false;
	m_mouseDown     = false;
	m_selectedPoint = e_noSelection;	
}

//	===========================================================================
CADSREditor::~CADSREditor()
{
	EXPONENT_CLASS_DESTRUCTION(CADSREditor);
}

//	===========================================================================
void CADSREditor::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}
	
	// Draw the background
	this->drawPrimaryImage(graphics, m_doDefaultDrawing);

	const CRect offset(0, 0, m_area.getWidth(), m_area.getHeight());
	const CRect drawToHereRelativeToMyTopLeft(0, 0, m_area.getWidth(), m_area.getHeight());

	// Draw the envelope polygon
	graphics.getMutablePen()->setColour(CAlphaColour(53, 140, 222, 255));
	graphics.getMutableBrush()->setColour(CAlphaColour(46, 82, 115, 255));
	graphics.beginAlphaDrawing(0.55);   
	graphics.fillPolygon(m_envelopePolygon);
	graphics.endAlphaDrawing(drawToHereRelativeToMyTopLeft, offset);
	graphics.drawAntiAliasedPolygon(m_envelopePolygon);

	// Draw the drag points if applicable
	if (m_mouseIsOver || m_mouseDown)
	{
		for (long i = 0; i < 4; i++)
		{
			if (m_selectedPoint == i)
			{
				graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_ORANGE);
			}
			else
			{
				graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_PURPLE);
			}
			graphics.drawRectangle(m_handles[i]);
		}
	}
}

//	===========================================================================
void CADSREditor::handleMouseLeavingArea(CMouseEvent &event)
{
	if (!m_mouseDown)
	{
		m_rootControl->unlockControl();
		m_mouseIsOver = false;
		this->update();
	}
}

//	===========================================================================
void CADSREditor::handleLeftButtonDown(CMouseEvent &event)
{
	// The mouse is now down
	m_mouseDown = true;

	// Default the selection so we can find a new one
	m_selectedPoint = e_noSelection;

	// Do we need to lock?
	if (m_selectedPoint != e_noSelection)
	{
		m_rootControl->lockControl(this);
	}

	// Store the mouse position
	const CPoint position = event.getMousePosition();

	// Find the point they clicked on
	for (long i = 0; i < 4; i++)
	{
		if (m_handles[i].pointIsInside(position))
		{
			m_selectedPoint = i;
			break;
		}
	}

	// Now update the graphics
	this->update();
}

//	===========================================================================
void CADSREditor::handleLeftButtonUp(CMouseEvent &event)
{
	// They released the mouse
	m_mouseDown = false;

	// Unlock
	m_rootControl->unlockControl();

	// Relock as necassary
	if (m_normalisedArea.pointIsInside(event.getMousePosition()))
	{
		m_rootControl->lockControl(this);
		m_mouseIsOver = true;
	}
	else
	{
		m_rootControl->unlockControl();
		m_mouseIsOver = false;
	}

	// Default the selection so we can find a new one
	m_selectedPoint = e_noSelection;

	// Redraw the graphics
	this->update();
}

//	===========================================================================
void CADSREditor::handleMouseMovement(CMouseEvent &event)
{
	// If the mouse is down then we need to handle it (assumming they have selected something
	if (m_mouseDown && m_selectedPoint != e_noSelection)
	{
		// Move the correct point
		switch(m_selectedPoint)
		{
			case 0:		this->moveAttack(event);		break;
			case 1:		this->moveDecay(event);			break;
			case 2:		this->moveSustain(event);		break;
			case 3:		this->moveRelease(event);		break;
		}

		// Notify the listener if we have one
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	else
	{
		if (m_normalisedArea.pointIsInside(event.getMousePosition()))
		{
			m_rootControl->lockControl(this);
			m_mouseIsOver = true;
		}
		else
		{
			m_rootControl->unlockControl();
			m_mouseIsOver = false;
		}
	}

	// Redraw the graphics
	this->update();
}

//	===========================================================================
void CADSREditor::moveAttack(CMouseEvent &event)
{
	// Get all of the points to edit
	CPoint *attackPoint  = m_envelopePolygon.getPoint(1);
	CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
	CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
	CPoint *releasePoint = m_envelopePolygon.getPoint(4);

	// Store the mouse position
	const CPoint position = event.getMousePosition();

	// Determine the gaps
	const long attackDecayGap    = decayPoint->getXPosition()   - attackPoint->getXPosition();
	const long releaseSustainGap = releasePoint->getXPosition() - sustainPoint->getXPosition();

	// We need to compute the left of the attack
	const long finalPosition   = CBounds::ensureRange(position.getXPosition(), CADSR_EDITOR_WINDOW_OFFSET, m_sizeOfQuarter);
	const long decayPosition   = finalPosition + attackDecayGap;
	const long sustainPosition = decayPosition + m_sizeOfQuarter;

	// Set the points
	attackPoint->setXPosition(finalPosition);
	decayPoint->setXPosition(decayPosition);
	sustainPoint->setXPosition(sustainPosition);
	releasePoint->setXPosition(sustainPosition + releaseSustainGap);

	// Setup the handles
	this->constructHandlePositions();

	// Store the value for the attack
	m_values[0] = (double)(finalPosition - CADSR_EDITOR_WINDOW_OFFSET) / (double)(m_sizeOfQuarter - CADSR_EDITOR_WINDOW_OFFSET);
}

//	===========================================================================
void CADSREditor::moveDecay(CMouseEvent &event)
{
	// Get all of the points to edit
	CPoint *attackPoint  = m_envelopePolygon.getPoint(1);
	CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
	CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
	CPoint *releasePoint = m_envelopePolygon.getPoint(4);

	// Store the mouse position
	const CPoint position = event.getMousePosition();

	// Determine the gaps
	const long releaseSustainGap = releasePoint->getXPosition() - sustainPoint->getXPosition();

	// We need to compute the left of the attack
	const long finalPosition   = CBounds::ensureRange(position.getXPosition(), attackPoint->getXPosition(), attackPoint->getXPosition() + m_sizeOfQuarter);
	const long sustainPosition = finalPosition + m_sizeOfQuarter;

	// Set the points
	decayPoint->setXPosition(finalPosition);
	sustainPoint->setXPosition(sustainPosition);
	releasePoint->setXPosition(sustainPosition + releaseSustainGap);

	// Setup the handles
	this->constructHandlePositions();

	// Store the value for the decay
	const long attackDecayGap = decayPoint->getXPosition() - attackPoint->getXPosition();
	m_values[1] = (double)attackDecayGap / (double)m_sizeOfQuarter;
}

//	===========================================================================
void CADSREditor::moveSustain(CMouseEvent &event)
{
	// Get all of the points to edit
	CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
	CPoint *sustainPoint = m_envelopePolygon.getPoint(3);

	// Store the mouse position
	const CPoint position = event.getMousePosition();

	// We need to compute the left of the attack
	const long finalPosition   = CBounds::ensureRange(position.getYPosition(), CADSR_EDITOR_WINDOW_OFFSET, CADSR_EDITOR_WINDOW_OFFSET + m_innerHeight);

	// Set the points
	decayPoint->setYPosition(finalPosition);
	sustainPoint->setYPosition(finalPosition);

	// Set the handles
	m_handles[1].setTop(decayPoint->getYPosition()   - CADSR_EDITOR_HANDLE_HALF_SIZE);
	m_handles[2].setTop(sustainPoint->getYPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE);

	// Store the new value for the sustain
	m_values[2] = 1.0 - (((double)finalPosition - CADSR_EDITOR_WINDOW_OFFSET) / (double)m_innerHeight);
}

//	===========================================================================
void CADSREditor::moveRelease(CMouseEvent &event)
{
	// Get all of the points to edit
	CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
	CPoint *releasePoint = m_envelopePolygon.getPoint(4);

	// Store the mouse position
	const CPoint position = event.getMousePosition();

	// We need to compute the left of the attack
	const long finalPosition   = CBounds::ensureRange(position.getXPosition(), sustainPoint->getXPosition(), sustainPoint->getXPosition() + m_sizeOfQuarter + CADSR_EDITOR_HANDLE_HALF_SIZE);

	// Set the points
	releasePoint->setXPosition(finalPosition);

	// Setup the handles
	this->constructHandlePositions();

	// Store the value for the decay
	const long releaseSustainGap = releasePoint->getXPosition() - sustainPoint->getXPosition();
	m_values[3] = (double)releaseSustainGap / (double)(m_sizeOfQuarter + CADSR_EDITOR_HANDLE_HALF_SIZE);
}

//	===========================================================================
void CADSREditor::constructHandlePositions()
{
	// Get all of the points to edit
	CPoint *attackPoint  = m_envelopePolygon.getPoint(1);
	CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
	CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
	CPoint *releasePoint = m_envelopePolygon.getPoint(4);

	// Set the handles
	m_handles[0].setLeft(attackPoint->getXPosition()  - CADSR_EDITOR_HANDLE_HALF_SIZE);
	m_handles[1].setLeft(decayPoint->getXPosition()   - CADSR_EDITOR_HANDLE_HALF_SIZE);
	m_handles[2].setLeft(sustainPoint->getXPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE);
	m_handles[3].setLeft(releasePoint->getXPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE);
}

//	===========================================================================
void CADSREditor::setValue(const long index, const double value)
{
	this->m_values[index] = value;

	switch(index)
	{
		case 0:	// Attack
			{
				// Store the position
				const long finalPosition = (long)((m_values[0] * (m_sizeOfQuarter - CADSR_EDITOR_WINDOW_OFFSET)) + CADSR_EDITOR_WINDOW_OFFSET);

				// Get all of the points to edit
				CPoint *attackPoint  = m_envelopePolygon.getPoint(1);
				CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
				CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
				CPoint *releasePoint = m_envelopePolygon.getPoint(4);

				// Determine the gaps
				const long attackDecayGap    = decayPoint->getXPosition()   - attackPoint->getXPosition();
				const long releaseSustainGap = releasePoint->getXPosition() - sustainPoint->getXPosition();

				// We need to compute the left of the attack
				const long decayPosition   = finalPosition + attackDecayGap;
				const long sustainPosition = decayPosition + m_sizeOfQuarter;

				// Set the points
				attackPoint->setXPosition(finalPosition);
				decayPoint->setXPosition(decayPosition);
				sustainPoint->setXPosition(sustainPosition);
				releasePoint->setXPosition(sustainPosition + releaseSustainGap);

				// Setup the handles
				this->constructHandlePositions();
			}
		break;
		case 1:	// Decay
			{
				// Get all of the points to edit
				CPoint *attackPoint  = m_envelopePolygon.getPoint(1);
				CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
				CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
				CPoint *releasePoint = m_envelopePolygon.getPoint(4);

				// Store the position
				const long finalPosition = (long)((m_values[1] * m_sizeOfQuarter) + attackPoint->getXPosition());

				// Store the gap
				const long releaseSustainGap = releasePoint->getXPosition() - sustainPoint->getXPosition();

				// Set the points
				decayPoint->setXPosition(finalPosition);
				sustainPoint->setXPosition(finalPosition + m_sizeOfQuarter);
				releasePoint->setXPosition((finalPosition + m_sizeOfQuarter) + releaseSustainGap);

				// Setup the handles
				this->constructHandlePositions();
			}
		break;
		case 2:	// Sustain
			{
				const long finalPosition = (long)(((1.0 - m_values[2]) * m_innerHeight) + CADSR_EDITOR_WINDOW_OFFSET);

				// Get all of the points to edit
				CPoint *decayPoint   = m_envelopePolygon.getPoint(2);
				CPoint *sustainPoint = m_envelopePolygon.getPoint(3);

				// Set the points
				decayPoint->setYPosition(finalPosition);
				sustainPoint->setYPosition(finalPosition);

				// Set the handles
				m_handles[1].setTop(decayPoint->getYPosition()   - CADSR_EDITOR_HANDLE_HALF_SIZE);
				m_handles[2].setTop(sustainPoint->getYPosition() - CADSR_EDITOR_HANDLE_HALF_SIZE);
			}
		break;
		case 3:	// Release
			{
				// Get all of the points to edit
				CPoint *sustainPoint = m_envelopePolygon.getPoint(3);
				CPoint *releasePoint = m_envelopePolygon.getPoint(4);

				// Final position
				const long finalPosition = (long)((m_values[3] * (m_sizeOfQuarter + CADSR_EDITOR_HANDLE_HALF_SIZE)) + sustainPoint->getXPosition());

				// Set the points
				releasePoint->setXPosition(finalPosition);
	
				// Setup the handles
				this->constructHandlePositions();
			}
		break;
	}
}